#!/usr/bin/env bash

if [[ -z $PHP_BINARY ]]; then
    export PHP_BINARY="$(which php)"
fi

set -o nounset
set -o errexit
set -o pipefail

declare no_symlinks='on'

# Linux/Mac abstraction
function get_realpath(){
    [[ ! -f "$1" ]] && return 1 # failure : file does not exist.
    [[ -n "$no_symlinks" ]] && local pwdp='pwd -P' || local pwdp='pwd' # do symlinks.
    echo "$( cd "$( echo "${1%/*}" )" 2>/dev/null; $pwdp )"/"${1##*/}" # echo result.
    return 0
}

# Set magic variables for current FILE & DIR
declare -r __FILE__=$(get_realpath ${BASH_SOURCE[0]})
declare -r __DIR__=$(dirname "$__FILE__")
declare -r __ENV_FILE__="$__DIR__/../../.env"

# Coloring/Styling helpers
esc=$(printf '\033')
reset="${esc}[0m"
blue="${esc}[34m"
green="${esc}[32m"
red="${esc}[31m"
bold="${esc}[1m"
warn="${esc}[41m${esc}[97m"

function printError(){
    >&2 echo -e "$@"
}

function promptYesOrNo(){
    declare prompt="$1"
    declare default=${2:-""}

    while true
        do
            read -p "$prompt" answer
                case $(echo "$answer" | `which awk` '{print tolower($0)}') in
                    y|yes)
                        echo 'y'
                        break
                        ;;
                    n|no)
                        echo 'n'
                        break
                        ;;
                    *)
                        if [ -z "$answer" ] && [ ! -z "$default" ] ; then
                            echo "$default"
                            break
                        fi
                        printError "Please enter y or n!"
                        ;;
        esac
    done
}

function swCommand(){
    ${PHP_BINARY} ${__DIR__}/../../bin/console "$@"
}

function banner(){
    echo -n "${blue}"
    cat "${__DIR__}/banner.txt"
    echo "${reset}"
}

function envFileDoesNotExists(){
    [ ! -f "${__ENV_FILE__}" ]
    return $?
}

function createEnvFile(){

    echo -e "\n--------------------------"
    echo -e "Database settings"
    echo -e "--------------------------\n"

    read -p "Enter your database host (default: 127.0.0.1): " DB_HOST
    DB_HOST=${DB_HOST:-"127.0.0.1"}

    read -p "Enter your database name (default: swcomposer): " DB_DATABASE
    DB_DATABASE=${DB_DATABASE:-swcomposer}

    read -p "Enter your database username (default: shopware): " DB_USERNAME
    DB_USERNAME=${DB_USERNAME:-shopware}

    read -p "Enter your database password (default: shopware): " DB_PASSWORD
    DB_PASSWORD=${DB_PASSWORD:-shopware}
    DB_PASSWORD=${DB_PASSWORD//\"/\\\"} # Escapes apostrophes

    read -p "Enter your database port number (default: 3306): " DB_PORT
    DB_PORT=${DB_PORT:-"3306"}

    echo -e "\n--------------------------"
    echo -e "Admin settings"
    echo -e "--------------------------\n"

    read -p "Admin username (default: demo): " ADMIN_USERNAME
    ADMIN_USERNAME=${ADMIN_USERNAME:-demo}

    read -p "Admin password (default: demo): " ADMIN_PASSWORD
    ADMIN_PASSWORD=${ADMIN_PASSWORD:-demo}

    read -p "Admin name (default: John Doe): " ADMIN_NAME
    ADMIN_NAME=${ADMIN_NAME:-"John Doe"}

    read -p "Admin email (default: demo@demo.com): " ADMIN_EMAIL
    ADMIN_EMAIL=${ADMIN_EMAIL:-"demo@demo.com"}

    echo -e "\n--------------------------"
    echo -e "Shop settings"
    echo -e "--------------------------\n"

    read -p "Enter your shop URL incl. protocol and path (default: http://shopware.example/path): " SHOP_URL
    SHOP_URL=${SHOP_URL:-http://shopware.example/path}

    IMPORT_DEMODATA=$(promptYesOrNo "Would you like to install demo data? (Y/n) " 'y')

    echo -e "# This file was generated by the shopware composer shell installer\n" > "${__ENV_FILE__}"
    echo -e "# Shop environment and database connection" >> "${__ENV_FILE__}"
    echo -e "SHOPWARE_ENV=\"dev\"" >> "${__ENV_FILE__}"

    echo -e "\n# The URL has priority over the other values, so only one parameter needs to be set in production environments" >> "${__ENV_FILE__}"
    echo -e "DATABASE_URL=\"mysql://${DB_USERNAME}:${DB_PASSWORD}@${DB_HOST}:${DB_PORT}/${DB_DATABASE}\"\n" >> "${__ENV_FILE__}"

    echo -e "# If e.g. the password contains special chars not allowed in a URL, you can define each parameter by itself instead" >> "${__ENV_FILE__}"
    echo -e "DB_HOST=\"${DB_HOST}\"" >> "${__ENV_FILE__}"
    echo -e "DB_DATABASE=\"${DB_DATABASE}\"" >> "${__ENV_FILE__}"
    echo -e "DB_USERNAME=\"${DB_USERNAME}\"" >> "${__ENV_FILE__}"
    echo -e "DB_PASSWORD=\"${DB_PASSWORD}\"" >> "${__ENV_FILE__}"
    echo -e "DB_PORT=\"${DB_PORT}\"" >> "${__ENV_FILE__}"

    echo -e "\n# Installation configuration (can be removed after installation)" >> "${__ENV_FILE__}"
    echo -e "ADMIN_EMAIL=\"$ADMIN_EMAIL\"" >> "${__ENV_FILE__}"
    echo -e "ADMIN_NAME=\"$ADMIN_NAME\"" >> "${__ENV_FILE__}"
    echo -e "ADMIN_USERNAME=\"$ADMIN_USERNAME\"" >> "${__ENV_FILE__}"
    echo -e "ADMIN_PASSWORD=\"$ADMIN_PASSWORD\"" >> "${__ENV_FILE__}"
    echo -e "SHOP_URL=\"$SHOP_URL\"\n" >> "${__ENV_FILE__}"
    echo -e "IMPORT_DEMODATA=$IMPORT_DEMODATA" >> "${__ENV_FILE__}"
}

function loadEnvFile(){
    if [ -f "${__ENV_FILE__}" ]; then
        echo "${green}Loading configuration settings from .env file${reset}"
        source "${__ENV_FILE__}"
        return
    fi
    echo "Could not load .env file"
    exit 1
}

function createEnvFileInteractive(){
    declare correct=0;

    while [[ ${correct} != 'y' ]]
        do
            createEnvFile
            echo -e "\n----------------------------------------------------------"
            echo -e "The following settings have been written to the .env file:"
            echo -e "----------------------------------------------------------\n"
            cat "${__ENV_FILE__}"
            echo -e "----------------------------------------------------------------\n"
            correct=$(promptYesOrNo "Is this information correct? (Y/n) " 'y')
    done
}

function createSymLinks(){
    echo "Creating symlinks in ${__DIR__}"
    cd "$__DIR__/../.."
    rm -rf  engine/Library
    mkdir -p engine/Library
    ln -s ../../vendor/shopware/shopware/engine/Library/ExtJs engine/Library/ExtJs
    ln -s ../../vendor/shopware/shopware/engine/Library/TinyMce engine/Library/TinyMce

    rm -rf themes/{package.json,Gruntfile.js,grunt-tasks}
    rm -rf themes/Frontend/{Bare,Responsive}
    rm -rf themes/Backend/ExtJs

    mkdir -p themes/{Frontend,Backend}

    ln -s ../../vendor/shopware/shopware/themes/Backend/ExtJs themes/Backend/ExtJs
    ln -s ../../vendor/shopware/shopware/themes/Frontend/Bare themes/Frontend/Bare
    ln -s ../../vendor/shopware/shopware/themes/Frontend/Responsive themes/Frontend/Responsive
    ln -s ../vendor/shopware/shopware/themes/package.json themes/package.json
    cp vendor/shopware/shopware/themes/Gruntfile.js themes/Gruntfile.js
    ln -s ../vendor/shopware/shopware/themes/grunt-tasks themes/grunt-tasks
}

