/**
 * @extends storeLocator.StaticDataFeed
 * @constructor
 */
function MedicareDataSource() {
    $.extend(this, new storeLocator.StaticDataFeed);

    var sourceCSV = '/fileadmin/redakteur/service/klinischezentren/klinischezentren_2024.csv';

    var that = this;
    $.get(sourceCSV, function (data) {
        that.setStores(that.parse_(data));
    });
}

/**
 * @const
 * @type {!storeLocator.FeatureSet}
 * @private
 */
MedicareDataSource.prototype.FEATURES_ = new storeLocator.FeatureSet(
    new storeLocator.Feature('Zentrum_Frau-YES', 'weiblicher Beckenboden'),
    new storeLocator.Feature('Zentrum_Mann-YES', 'männliche Inkontinenz'),
    new storeLocator.Feature('Zentrum_Brustrekonstruktion-YES', 'Brustrekonstruktion'),
    new storeLocator.Feature('Zentrum_Hiatushernie-YES', 'Hiatushernie')
);

/**
 * @return {!storeLocator.FeatureSet}
 */
MedicareDataSource.prototype.getFeatures = function () {
    return this.FEATURES_;
};

/**
 * @private
 * @param {string} csv
 * @return {!Array.<!storeLocator.Store>}
 */
MedicareDataSource.prototype.parse_ = function (csv) {
    var stores = [];
    var rows = csv.split('\n');
    var headings = this.parseRow_(rows[0]);

    for (var i = 1, row; row = rows[i]; i++) {
        row = this.toObject_(headings, this.parseRow_(row));

        var features = new storeLocator.FeatureSet;
        features.add(this.FEATURES_.getById('Zentrum_Frau-' + row.Zentrum_Frau));
        features.add(this.FEATURES_.getById('Zentrum_Mann-' + row.Zentrum_Mann));
        features.add(this.FEATURES_.getById('Zentrum_Brustrekonstruktion-' + row.Zentrum_Brustrekonstruktion));
        features.add(this.FEATURES_.getById('Zentrum_Hiatushernie-' + row.Zentrum_Hiatushernie));

        var position = new google.maps.LatLng(row.Lat, row.Lng);
        var locality = this.join_([row.PLZ, row.Ort], ' ');

        var icon;
        if (row.Zentrum_Mann) {
            icon = "/typo3conf/ext/seragwiessner_sitepackage/Resources/Public/Maps/marker/p/mann.png";
            //console.log('row.Zentrum_Mann');
        }
        if (row.Zentrum_Frau) {
            icon = "/typo3conf/ext/seragwiessner_sitepackage/Resources/Public/Maps/marker/p/frau.png";
            //console.log('row.Zentrum_Frau');
        }
        if (row.Zentrum_Mann && row.Zentrum_Frau) {
            icon = "/typo3conf/ext/seragwiessner_sitepackage/Resources/Public/Maps/marker/p/beides.png";
            //console.log('row.Zentrum_Mann&&row.Zentrum_Frau');
        }
        if (row.Zentrum_Brustrekonstruktion) {
            icon = "/typo3conf/ext/seragwiessner_sitepackage/Resources/Public/Maps/marker/p/brust.png";
            //console.log('row.Zentrum_Brustrekonstruktion');
        }
        if (row.Zentrum_Frau && row.Zentrum_Brustrekonstruktion) {
            icon = "/typo3conf/ext/seragwiessner_sitepackage/Resources/Public/Maps/marker/p/frau-brust.png";
            //console.log('row.Zentrum_Mann&&row.Zentrum_Frau');
        }
        if (row.Zentrum_Hiatushernie) {
            icon = "/typo3conf/ext/seragwiessner_sitepackage/Resources/Public/Maps/marker/p/hernie.png";
        }


        var store = new storeLocator.Store(i, position, features, icon, {
            klinik1: row.Klinik1,
            klinik2: row.Klinik2,
            abteilung1: row.Abteilung1,
            abteilung2: row.Abteilung2,
            arzt: row.Arzt,
            ort: row.Ort,
            plz: row.PLZ,
            adresse: this.join_([row.Straße, locality], '<br/>'),
            telefon: row.Telefon,
            telefax: row.Fax,
            email: row.Email,
            filtertext: row.Filtertext,
            web: '<a href="' + row.WWW + '" target="_blank">Website ansehen</a>'
        });
        stores.push(store);
    }
    return stores;
};

/**
 * Joins elements of an array that are non-empty and non-null.
 * @private
 * @param {!Array} arr array of elements to join.
 * @param {string} sep the separator.
 * @return {string}
 */
MedicareDataSource.prototype.join_ = function (arr, sep) {
    var parts = [];
    for (var i = 0, ii = arr.length; i < ii; i++) {
        arr[i] && parts.push(arr[i]);
    }
    return parts.join(sep);
};

/**
 * Very rudimentary CSV parsing - we know how this particular CSV is formatted.
 * IMPORTANT: Don't use this for general CSV parsing!
 * @private
 * @param {string} row
 * @return {Array.<string>}
 */
MedicareDataSource.prototype.parseRow_ = function (row) {
    // Strip leading quote.
    if (row.charAt(0) == '"') {
        row = row.substring(1);
    }
    // Strip trailing quote. There seems to be a character between the last quote
    // and the line ending, hence 2 instead of 1.
    if (row.charAt(row.length - 2) == '"') {
        row = row.substring(0, row.length - 2);
    }

    // row = row.split('","');
    // soll beim tab splitten!
    row = row.split('\t', -1);

    return row;
};

/**
 * Creates an object mapping headings to row elements.
 * @private
 * @param {Array.<string>} headings
 * @param {Array.<string>} row
 * @return {Object}
 */
MedicareDataSource.prototype.toObject_ = function (headings, row) {
    var result = {};
    for (var i = 0, ii = row.length; i < ii; i++) {
        result[headings[i]] = row[i];
    }
    return result;
};


function initialize() {

    var mapDiv = document.getElementById('map-canvas'),
        contentDiv = document.getElementById('content'),
        panelDiv = document.getElementById('panel'),
        naila = new google.maps.LatLng(50.326807, 11.690567), // Serag Naila
        center = new google.maps.LatLng(50.326807, 4.690567); // hierhin wird zentriert

    window.innerHeight = window.innerHeight || document.documentElement.clientHeight;
    contentDiv.style.minHeight = window.innerHeight - 200 + "px";

    var myMapOptions = {
        center: naila,
        zoom: 6,
        scrollwheel: false,
        navigationControl: true,
        panControl: true,
        mapTypeControl: false,
        scaleControl: true,
        draggable: true,
        zoomControlOptions: {
            style: google.maps.ZoomControlStyle.LARGE,
            position: google.maps.ControlPosition.RIGHT_CENTER
        },
        streetViewControl: true,
        styles: [
            {
                "featureType": "water",
                "elementType": "labels",
                "stylers": [
                    {"visibility": "off"}
                ]
            }, {
                "featureType": "road",
                "elementType": "labels",
                "stylers": [
                    {"visibility": "off"}
                ]
            }, {
                "featureType": "poi",
                "elementType": "labels",
                "stylers": [
                    {"visibility": "off"}
                ]
            }, {
                "featureType": "administrative",
                "elementType": "labels",
                "stylers": [
                    {"visibility": "off"}
                ]
            }, {
                "featureType": "landscape",
                "stylers": [
                    {"visibility": "simplified"},
                    {"saturation": -100},
                    {"gamma": 2.86}
                ]
            }, {
                "featureType": "road.local",
                "elementType": "geometry",
                "stylers": [
                    {"visibility": "off"}
                ]
            }, {
                "featureType": "road.arterial",
                "elementType": "geometry",
                "stylers": [
                    {"saturation": -100},
                    {"gamma": 1.61},
                    {"visibility": "simplified"}
                ]
            }, {
                "featureType": "road.highway",
                "elementType": "geometry",
                "stylers": [
                    {"visibility": "simplified"},
                    {"saturation": -100},
                    {"gamma": 2.75}
                ]
            }, {
                "featureType": "poi",
                "stylers": [
                    {"visibility": "off"}
                ]
            }, {
                "featureType": "water",
                "stylers": [
                    {"saturation": -96}
                ]
            }, {
                "featureType": "administrative.country",
                "elementType": "labels",
                "stylers": [
                    {"visibility": "on"},
                    {"saturation": -100},
                    {"gamma": 3.6}
                ]
            }
        ]
    };


    var map = new google.maps.Map(mapDiv, myMapOptions);

    if ( window.innerWidth > '768'){
        map.panBy(-300, 0);
    }

    var data = new MedicareDataSource;
    var view = new storeLocator.View(map, data, {
        geolocation: false,
        features: data.getFeatures()
    });

    new storeLocator.Panel(panelDiv, {
        view: view,
        directions: false,
        locationSearch: false,
        featureFilter: true,
        zoomLink: false,
        directionsLink: false,
        streetViewLink: false
    });
}

google.maps.event.addDomListener(window, 'load', initialize);

$(window).on('resize', function () {
    google.maps.event.trigger(map, "resize");
})

