<?php

namespace App\Http\Controllers;

use App\Animal;
use App\AnimalAviary;
use App\AnimalPair;
use App\Championship;
use App\ChampionshipParticipation;
use App\Exhibition;
use App\Exhibitor;
use App\Helper\PriceHelper;
use App\Helper\RegistrationHelper;
use App\Notifications\MeldebogenComplete;
use App\Notifications\RegistrationComplete;
use App\Notifications\TelegramNotification;
use App\Registration;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class RegistrationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }


    /**
     * @param Registration $registration
     */
    public static function send(Registration $registration)
    {
        $user = Auth::user();

        $exhibitor = Exhibitor::find($registration->exhibitorId);

        if ($user->id != $exhibitor->userId) return redirect()->back();

        Auth::user()->notify(new MeldebogenComplete($registration));
        Auth::user()->notify(new TelegramNotification(($registration->locked ? 'Meldebogen geändert' : 'Meldebogen erhalten') . ' von ' . $exhibitor->getFullname()));


        $registration->locked = 1;
        $registration->save();


        return redirect()->action('ExhibitionController@index');
    }


    /**
     * Show the form for creating a new resource.
     *
     * @param Exhibition $exhibition
     * @return \Illuminate\Http\Response
     */
    public function create(Exhibition $exhibition)
    {
        $prices = PriceHelper::getPrices();

        $exhibitors = RegistrationHelper::getNonRegisterExhibitorsForExhibition($exhibition->id);

        return view('registration.create', compact('exhibition', 'exhibitors', 'prices'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $registration = new Registration();

        $exhibitor = Exhibitor::find($request->exhibitor);

        if (Auth::id() != $exhibitor->userId) return redirect()->back();

        if ($exhibitor->type > 1 && $request->judge == "on") {
            return back()->withErrors('Jugendlicher Preisrichter endeckt.');
        }


        $registration->exhibitionId = $request->exhibition;
        $registration->userId = Auth::id();
        $registration->exhibitorId = $request->exhibitor;
        $registration->is_judge = $request->judge == "on" ? true : false;


        $registration->save();

        return redirect()->action('RegistrationController@show', ['registration' => $registration->id]);

    }

    /**
     * Display the specified resource.
     *
     * @param Request $request
     * @param Registration $anmeldung
     * @return view
     */
    public function show(Request $request, Registration $anmeldung)
    {


        $registration = $anmeldung;

        $user = Auth::user();

        $exhibitor = Exhibitor::find($registration->exhibitorId);

        if ($user->id != $exhibitor->userId) return redirect()->back();


        $exhibition = Exhibition::find($registration->exhibitionId);


        $championships = Championship::where('exhibitionId', $exhibition->id)->get();

        $animals = Animal::where('exhibitionId', $registration->exhibitionId)->where('exhibitorId', $registration->exhibitorId)->orderBy('id', 'asc')->get();
        $animalPairs = AnimalPair::where('exhibitionId', $registration->exhibitionId)->where('exhibitorId', $registration->exhibitorId)->orderBy('id', 'asc')->get();
        $animalAviaries = AnimalAviary::where('exhibitionId', $registration->exhibitionId)->where('exhibitorId', $registration->exhibitorId)->orderBy('id', 'asc')->get();


        $count = [
            "priceCatalog" => !($registration->is_judge or $exhibitor->type != 1),
            "priceSpecial" => $registration->amount_special,
            "pricePrize" => $registration->amount_prize,
            "priceSpecialBonus" => $registration->amount_specialBonus,
            "priceAviaries" => $animalAviaries->count(),
            "pricePairs" => $animalPairs->count(),
            "priceSingle" => $exhibitor->type == 1 ? $animals->count() : 0,
            "priceSingleJunior" => $exhibitor->type == 2 ? $animals->count() : 0,
            "priceNewBreed" => 0
        ];


        $championShipFee = 0;

        foreach ($championships as $championship) {

            if ($championship->fee != 0) {

                $championShipFee += ChampionshipParticipation::where('exhibitorId', $exhibitor->id)
                        ->where('exhibitionId', $exhibition->id)
                        ->where('championshipId', $championship->id)
                        ->count() * $championship->fee;
            }
        }

        $last_color = $request->session()->get('last_color');
        $last_breed = $request->session()->get('last_breed');
        $last_position = $request->session()->get('last_position');
        $request->session()->put('last_position', null);


        $breed_options = RegistrationHelper::getSelect2Breeds();
        $color_options = RegistrationHelper::getSelect2Colors();


        $prizes = PriceHelper::getPrizes();


        $context = compact('registration',
            'exhibition',
            'exhibitor',
            'prizes',
            'breed_options',
            'color_options',
            'animals',
            'animalPairs',
            'animalAviaries',
            'count',
            'last_color',
            'last_breed',
            'last_position',
            'championships',
            'championShipFee'
        );


        return view('registration.show', $context);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Registration $anmeldung)
    {

    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param Registration $anmeldung
     * @return void
     */
    public function update(Request $request, Registration $anmeldung)
    {

        foreach ($request->all() as $k => $v) {
            if (in_array($k, ['_token', '_method', 'exhibitorType'])) {
                continue;
            }
            $anmeldung->$k = $v;
        }
        $anmeldung->save();

        return redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
